      program spins                                                 
c
c---- Generates the Rumer spin functions; transforms b/n
c---- Rumer, Kotani and Serber spin functions; calculates the
c---- symmetric group representation matrices in the Rumer, 
c---- Kotani and Serber spin bases.
c
c---- Works up to n = 20, spin = 1 (16796 spin eigenfunctions)
c---- Note: nmax and ifnsm can be increased, but not beyond
c---- nmax = 31 (4-byte integers are used to hold the bit-coded
c---- spin factors and bit 0 is not used)
c
c---- Current version uses LINPACK routines 
c---- dspfa (daxpy, dswap, idamax), dspdi (dswap, daxpy, ddot, dcopy), 
c---- dtrdi (daxpy, dscal), dgefa (daxpy, dscal, idamax),
c---- dgesl (daxpy, ddot)
c
c---- Uses dynamic arrays and etime; save in getfld 
c
      parameter (nmax = 20, ifnsm = 16796)
c
      implicit real*8 (a-h, o-z)
      common ustart, sstart ! etime
      real*4 ustart, sstart ! etime
      character*(nmax) lt(ifnsm) 
      integer*2 lts(ifnsm) 
      dimension iperm(nmax), iiperm(nmax)
c       
      character*(nmax) srb(:)
      integer*4 rd(:), rdp(:)
      integer*2 rf(:), rs(:)
      allocatable srb, rd, rdp, rf, rs, 
     .          rr(:), rrinv(:), trk(:), tkr(:),
     .          vr(:), vk(:), vs(:), wsq(:), 
     .          tks(:), work(:), kw(:), is(:), 
     .          kpvt(:), cs(:), ct(:)
c
      character*81 field, lineout
      character*10 intype, trtype, repbas
      integer inp /5/, iout /6/, lcount /0/
      logical neven, newrun, vectin, reprin, permin,
     .        rumovr, tranrk, trankr, spaths, tranks, reprr,
     .        reprk, reprs, repout, vecout, permeq,
     .        prnlt /.false./, prnks /.false./, prndet /.false./,
     .        prnovr /.false./, prntr /.false./, aalloc /.false./
c
      call timer(ustart, sstart) ! etime
c
      write (iout, '(1x, ''    //////   //////    //'', 
     .                   ''  //   //   //////  version 1.2'')')
      write (iout, '(1x, ''  ///       //    //  // '',
     .                   '' ///  //  ///      PBK 1993    '')')
      write (iout, '(1x, ''   ////    ///////   //  '',
     .                   ''// / //    ////   Bristol      '')')            
      write (iout, '(1x, ''     ///  //        //  /'',
     .                   ''/  ///       ///               '')')             
      write (iout, '(1x, ''//////   //        //  //'',
     .                   ''   //   //////                 '')')            
c
 10   call getfld(inp, iout, field, lcount)
c
      if (index(field, 'electrons').eq.1)      go to 100
      if (index(field, 'spin').eq.1)           go to 200
      if (index(field, 'input').eq.1.or.
     .    index(field, 'permutation').eq.1)    go to 300
      if (index(field, 'transform').eq.1)      go to 400
      if (index(field, 'representation').eq.1) go to 300
      if (index(field, 'print').eq.1)          go to 500
      write (6, *) field
      call abend(iout, 'Invalid command$', lcount)
c
c---- Read in number of electrons
c
 100  call getfld(inp, iout, field, lcount)
      call str2i(field, n, iout, lcount)
      if (n.gt.nmax) call abend(iout, 'Too many electrons$', lcount)
      if (n.le.0) call abend(iout,
     .            'Zero or negative number of electrons$', lcount)
c---- Default spin
      neven = (n/2)*2.eq.n
      if (neven) then
        spin = 0d0
      else
        spin = 0.5d0
      endif
      newrun = .true.
      permin = .false.
      reprin = .false.
      go to 10
c
c---- Input spin if specified
c
 200  if (n.eq.0) call abend(iout, 'Command out of order$', lcount)
      call getfld(inp, iout, field, lcount)
      call str2f(field, spin, iout, lcount)
c---- Check if spin is compatible with number of electrons
      if (neven) then
        jspin = 0
        do 210 i = 0, n/2
          ispin = i
          if (spin.eq.dfloat(i)) go to 230
 210    continue
      else
        jspin = 2
        do 220 i = 1, n, 2
          ispin = i
          if (spin.eq.dfloat(i)/2d0) go to 230
 220    continue
      endif
      call abend(iout, 'Spin incompatible with number of electrons$',
     .           lcount)
 230  newrun = .true.
      reprr = .false.
      reprk = .false.
      reprs = .false.
      repout = .false.
      go to 10
c
c---- Read in input vector or permutation, or calculate representation matrix
c
 300  if (n.eq.0.or.
     .  index(field, 'representation').eq.1.and..not.permin)
     .  call abend(iout, 'Command out of order$', lcount)
      if (newrun) then
c------ Generate leading terms -----------------------------------------
        call leadt(n, spin, npairs, ifns, lt, lts, prnlt, iout)
        if (jspin.eq.0) then
          write (lineout, '(''System of '', i2,
     .                      '' electrons with spin '', i2,
     .                      '' (f^N_S = '', i5, '')$'' )')
     .    n, ispin, ifns
        else
          write (lineout, '(''System of '', i2,
     .                      '' electrons with spin '', i2, ''/'', i1,
     .                      '' (f^N_S = '', i5, '')$'' )')
     .     n, ispin, jspin, ifns
        endif
 301    i = index(lineout, '  ')
        j = index(lineout, '$')
        if (i.gt.0.and.i.lt.j) then
          do 302 k = i + 1, j - 1
 302      lineout(k:k) = lineout(k + 1:k + 1)
          go to 301
        endif
        write (iout, '(/1x, a)') lineout(1:j - 1)
c------ If arrays are allocated, deallocate them 
        if (aalloc) then 
          deallocate(srb, rd, rdp, rf, rs, rr, rrinv, trk, tkr,  
     .               vr, vk, vs, wsq, tks, work, kw, is, kpvt,
     .               cs, ct, stat = ierr)
          if (ierr.ne.0) call abend(iout,
     .       'Cannot free dynamically allocated memory$', 0)
        endif
c------ Allocate arrays     
        ndet = 2**npairs
        allocate(srb(ifns), rd(ifns*ndet), rdp(ifns*ndet),
     .           rf(ifns*n), rs(ifns*ndet),
     .           rr(ifns*(ifns + 1)/2), rrinv(ifns*(ifns + 1)/2),
     .           trk(ifns*(ifns + 1)/2), tkr(ifns*(ifns + 1)/2),
     .           vr(ifns*ifns), vk(ifns*ifns), vs(ifns*ifns),
     .           wsq(ifns*ifns), tks(ifns*ifns),
     .           work(ifns), kw(ifns + 1), is(ifns), kpvt(ifns),
     .           cs(ifns), ct(ifns), stat = ierr)
        if (ierr.ne.0) call abend(iout,
     .     'Not enough memory for data$', 0)
        aalloc = .true.
c------ Calculate addressing vector for sym/ut matrices ----------------
        kw(1) = 0
        do 310 i = 2, ifns + 1
 310    kw(i) = kw(i - 1) + i - 1
        newrun = .false.
        vectin = .false.
        rumovr = .false.
        tranrk = .false.
        trankr = .false.
        spaths = .false.
        tranks = .false.
      endif
      if (index(field, 'input').eq.1) then
        call getbas(intype, inp, iout, field, lcount, .true.)
        if (index(intype, 'Rumer').eq.1.and..not.rumovr) then
c-------- Form Rumer functions -----------------------------------------
          call fpairs(n, ifns, npairs, lt, rf)
c-------- Expand the Rumer functions in terms of signed spin factors ---
          call fdeter(n, ifns, npairs, ndet, rf, rd, rs, prndet, iout)
c-------- Calculate the overlap matrix b/n the Rumer functions ---------
          call rover(ifns, npairs, ndet, rd, rs, rr, rrinv, kpvt, 
     .               work, prnovr, iout)
          rumovr = .true.
        elseif (index(intype, 'Serber').eq.1.and..not.spaths) then
c-------- Map Serber to Kotani functions -------------------------------
          call fsrb(n, ifns, lt, srb, prnks, iout)
          spaths = .true.
        endif
c------ Read in coefficients -------------------------------------------
        do 320 i = 1, ifns
          call getfld(inp, iout, field, lcount)
          call str2f(field, cs(i), iout, lcount)
 320    continue
        if (index(intype, 'Rumer_VB').eq.1) then
          do 325 i = 1, ifns
 325      cs(i) = lts(i)*cs(i)
          write (iout, '(/1x, ''(Rumer_VB input converted to Rumer)'')')
          intype = 'Rumer$'
        endif
c------ Renormalize input coefficients ---------------------------------
        call renorm(ifns, cs, rr, intype, kw)
        write (iout, '(/1x, ''Renormalized input spin function '',
     .        ''coefficients ('', a, '' spin basis):''/)')
     .        intype(1:index(intype, '$') - 1)
        call cout(iout, intype, n, ifns, npairs, rf, lt, srb,
     .            cs, rr, rrinv, work, kw, .true.)
        vectin = .true.
        vecout = .false.
      elseif (index(field, 'permutation').eq.1) then
        do 340 i = 1, n
          call getfld(inp, iout, field, lcount)
          call str2i(field, iperm(i), iout, lcount)
c-------- Check for repetition of indices ------------------------------
          do 330 j = 1, i - 1
            if (iperm(i).eq.iperm(j))
     .        call abend(iout, 'Invalid permutation$', lcount)
 330      continue
 340    continue
        write (iout, '(/1x, ''Permutation: ''/)')
        write (iout, '(1x, 12(i3, 1x))') (iperm(i), i = 1, n)
c------ Invert permutation ---------------------------------------------
        do 350 i = 1, n
        do 350 j = 1, n
          if (iperm(j).eq.i) then
            iiperm(i) = j
            go to 350
          endif
 350    continue
        permeq = .true.
        do 360 i = 1, n 
          if (iperm(i).ne.iiperm(i)) then
            iperm(i) = iiperm(i)
            permeq = .false.
          endif 
 360    continue  
        write (iout, '(/1x, ''Inverse permutation: ''/)')
        write (iout, '(1x, 12(i3, 1x))') (iperm(i), i = 1, n)
        permin = .true.
        reprr = .false.
        reprk = .false.
        reprs = .false.
        repout = .false.
        vecout = .false.
      else
        call getbas(repbas, inp, iout, field, lcount, .false.)
        reprin = .true.
        repout = .false.
      endif
c---- Permute vector if vector and permutation available or ------------
c---- print/calculate S_N representation matrix if requested -----------
      if ((vectin.or.reprin).and.permin) then
        if (.not.rumovr) then
c-------- Form Rumer functions -----------------------------------------
          call fpairs(n, ifns, npairs, lt, rf)
c-------- Expand the Rumer functions in terms of signed spin factors ---
          ndet = 2**npairs
          call fdeter(n, ifns, npairs, ndet, rf, rd, rs, prndet, iout)
c-------- Calculate the overlap matrix b/n the Rumer functions ---------
          call rover(ifns, npairs, ndet, rd, rs, rr, rrinv, kpvt, 
     .               work, prnovr, iout)
          rumovr = .true.
        endif
        if (index(intype // repbas, 'Kotani').gt.0.or.
     .      index(intype // repbas, 'Serber').gt.0) then
          if (.not.tranrk) then
            call rumkot(ifns, rr, trk, work, kw, prntr, iout)
            tranrk = .true.
          endif
          if (.not.trankr) then
            call kotrum(ifns, trk, tkr, wsq, prntr, iout)
            trankr = .true.
          endif
        endif
        if (index(intype // repbas, 'Serber').gt.0.and..not.tranks) then
          if (.not.spaths) then
c---------- Map Serber to Kotani functions -----------------------------
            call fsrb(n, ifns, lt, srb, prnks, iout)
            spaths = .true.
          endif
          call kotsrb(n, ifns, srb, tks, is, prntr, iout)
          tranks = .true.
        endif
        if (.not.reprr) then
          call permr(n, ifns, ndet, npairs, iperm, rd, rdp, rs,
     .               rrinv, wsq, vr, kw)
          reprr = .true.
        endif
        if (index(intype, 'Rumer').eq.1.and.vectin.and..not.vecout) then
          if (permeq) then
            call matvec(ifns, ct, vr, cs, kw, 1)  
          else
            do 370 i = 1, ifns*ifns
 370        wsq(i) = vr(i)   
            call dgefa(wsq, ifns, ifns, kpvt, info)          
            if (info.ne.0) call abend(iout,
     .        '(''V^R(P) is singular. Cannot continue ...$'')', 0)
            call dgesl(wsq, ifns, ifns, kpvt, cs, 0)
          endif
        endif
        if (index(intype // repbas, 'Kotani').gt.0) then
          if (.not.reprk) then
            call permk(ifns, vr, trk, tkr, vk, work, kw)
            reprk = .true.
          endif
          if (index(intype, 'Kotani').eq.1.and.vectin.and..not.vecout)
c---------- (new)
     .      call matvec(ifns, ct, vk, cs, kw, 2)
        endif
        if (index(intype // repbas, 'Serber').gt.0) then
          if (.not.reprk) then
            call permk(ifns, vr, trk, tkr, vk, work, kw)
            reprk = .true.
          endif
          if (.not.reprs) then
            call perms(ifns, vk, tks, vs, work)
            reprs = .true.
          endif
          if (index(intype, 'Serber').eq.1.and.vectin.and..not.vecout)
c---------- (new)
     .      call matvec(ifns, ct, vs, cs, kw, 2)
        endif
        if (reprin.and..not.repout) then
          write (iout, '(/1x, ''Symmetric group representation matrix'',
     .          '' in the '', a, '' spin basis V^'', a, ''(P):''/)')
     .          repbas(1:index(repbas, '$') - 1), repbas(1:1)
          if (index(repbas, 'Rumer').eq.1) then
            call fulout(iout, vr, vr, ifns, ifns, 2, 5)
          elseif (index(repbas, 'Kotani').eq.1) then
            call fulout(iout, vk, vk, ifns, ifns, 2, 5)
          elseif (index(repbas, 'Serber').eq.1) then
            call fulout(iout, vs, vs, ifns, ifns, 2, 5)
          endif
          repout = .true.
        endif
        if (vectin.and..not.vecout) then
          write (iout, '(/1x, ''Permuted spin function coefficients ('',
     .       a, '' spin basis):''/)') intype(1:index(intype, '$') - 1)
          call cout(iout, intype, n, ifns, npairs, rf, lt, srb,
     .      cs, rr, rrinv, work, kw, .true.)
          vecout = .true.
        endif
      endif
      go to 10
c
c---- Get transformation type and perform it
c
 400  if (.not.vectin) call abend(iout, 'Command out of order$', lcount)
      call getbas(trtype, inp, iout, field, lcount, .false.)
      write (iout, '(/1x, a, '' to '', a, '' transformation'')')
     .      intype(1:index(intype, '$') - 1),
     .      trtype(1:index(trtype, '$') - 1)
      if (intype.eq.trtype) go to 10
      if (index(intype // trtype, 'Rumer').ne.0) then
        if (.not.rumovr) then
c-------- Form Rumer functions -----------------------------------------
          call fpairs(n, ifns, npairs, lt, rf)
c-------- Expand the Rumer functions in terms of signed spin factors ---
          ndet = 2**npairs
          call fdeter(n, ifns, npairs, ndet, rf, rd, rs, prndet, iout)
c-------- Calculate the overlap matrix b/n the Rumer functions ---------
          call rover(ifns, npairs, ndet, rd, rs, rr, rrinv, kpvt, 
     .               work, prnovr, iout)
          rumovr = .true.
        endif
        if (.not.tranrk) then
          call rumkot(ifns, rr, trk, work, kw, prntr, iout)
          tranrk = .true.
        endif
        if (index(intype, 'Rumer').eq.1.and..not.trankr) then
          call kotrum(ifns, trk, tkr, wsq, prntr, iout)
          trankr = .true.
        endif
      endif
      if (index(intype // trtype, 'Serber').ne.0.and..not.tranks) then
        if (.not.spaths) then
c-------- Map Serber to Kotani functions -------------------------------
          call fsrb(n, ifns, lt, srb, prnks, iout)
          spaths = .true.
        endif
        call kotsrb(n, ifns, srb, tks, is, prntr, iout)
        tranks = .true.
      endif
      if (index(intype, 'Rumer').eq.1) then
        call matvec(ifns, ct, tkr, cs, kw, 3)
        if (index(trtype, 'Serber').eq.1)
     .    call matvec(ifns, ct, tks, cs, kw, 2)
      elseif (index(intype, 'Kotani').eq.1) then
        if (index(trtype, 'Rumer').eq.1) then
          call matvec(ifns, ct, trk, cs, kw, 4)
        elseif (index(trtype, 'Serber').eq.1) then
          call matvec(ifns, ct, tks, cs, kw, 2)
        endif
      elseif (index(intype, 'Serber').eq.1) then
        call matvec(ifns, ct, tks, cs, kw, 1)
        if (index(trtype, 'Rumer').eq.1)
     .    call matvec(ifns, ct, trk, cs, kw, 4)
      endif
      write (iout, '(/1x, ''Transformed spin function coefficients ('',
     .      a, '' spin basis):''/)') trtype(1:index(trtype, '$') - 1)
      call cout(iout, trtype, n, ifns, npairs, rf, lt, srb,
     .          cs, rr, rrinv, work, kw, .true.)
      intype = trtype
      go to 10
c
 500  call getfld(inp, iout, field, lcount)
      if (index(field, 'leading_terms').eq.1) then
        prnlt = .true.
      elseif (index(field, 'kotani_and_serber_paths').eq.1) then
        prnks = .true.
      elseif (index(field, 'rumer_functions').eq.1) then
        prndet = .true.
      elseif (index(field, 'rumer_overlaps').eq.1) then
        prnovr = .true.
      elseif (index(field, 'transformations').eq.1) then
        prntr = .true.
      elseif (index(field, 'everything').eq.1.or.
     .        index(field, 'all').eq.1) then
        prnlt =  .true.
        prnks =  .true.
        prndet = .true.
        prnovr = .true.
        prntr =  .true.
      elseif (index(field, 'nothing').eq.1) then
        prnlt =  .false.
        prnks =  .false.
        prndet = .false.
        prnovr = .false.
        prntr =  .false.
      else
        call abend(iout, 'Unknown keyword$', lcount)
      endif
      go to 10
c
      stop
      end
c
c----------------------------------------------------------------------
c
      subroutine getfld(inp, iout, field, lcount)
c       
      common ustart, sstart ! etime
      real*4 ustart, sstart ! etime
      character*(*) field
      character*80 linein
      character*4 blank /' ,;/'/
      integer i1 /1/
      logical anycmd /.false./, comment
      save linein, i1, anycmd
c
      if (i1.eq.1) then
 5      read (inp, '(a)', end = 1000) linein
        lcount = lcount + 1
        if (linein(1:1).eq.'%') then
          call linout(iout, linein, comment)
          comment = .true.
          go to 5
        else
          comment = .false.
        endif
        call lcase(linein)
      endif
 10   if (index(blank, linein(i1:i1)).ne.0) then
        if (i1.lt.len(linein)) then
          i1 = i1 + 1
        else
 15       read (inp, '(a)', end = 1000) linein
          lcount = lcount + 1
          if (linein(1:1).eq.'%') then
            call linout(iout, linein, comment)
            comment = .true.
            go to 15
          else
            comment = .false.
          endif
          call lcase(linein)
          i1 = 1
        endif
        go to 10
      endif
c
      i2 = i1 + 1
 20   if (index(blank, linein(i2:i2)).eq.0.and.
     .   i2.lt.len(linein)) then
        i2 = i2 + 1
        go to 20
      endif
      if (i2.ne.len(linein)) i2 = i2 - 1
      anycmd = .true.
c
      field(1:(i2 + 1 - i1) + 1) = linein(i1:i2) // '$'
      i1 = i2 + 1
      if (i1.gt.len(linein)) i1 = 1
c
      return
c
 1000 if (.not.anycmd) then
        call abend(iout, 'Nothing to do: Blank input file ...$', 0)
      else
        call timer(uend, send) ! etime (this one + 3 write stmts)
        write (iout, '(/1x, ''User time:   '', f10.2, '' sec'')')
     .        uend - ustart
        write (iout, '( 1x, ''System time: '' , f10.2, '' sec'')')
     .        send - sstart
        write (iout, '( 1x, ''Total time:  '' , f10.2, '' sec'')')
     .        uend - ustart + send - sstart
        stop '--- spins completed.'
      endif
c
      end
c
c----------------------------------------------------------------------
c
      subroutine linout(iout, line, comment)
c
      character*(*) line
      logical comment
c
      do 10 i = len(line), 1, -1
        if (line(i:i).ne.' ') then
          if (comment) then
            write (iout, '(1x, a)') line(1:i)
          else
            write (iout, '(/1x, a)') line(1:i)
          endif
          return
        endif
 10   continue
c
      end
c
c----------------------------------------------------------------------
c
      subroutine lcase(string)
      character*(*) string
c
c---- Convert to lower case (use ASCII table)
c
      do 10 i = 1, len(string)
         mark = ichar(string(i:i))
         if (mark.ge.65.and.mark.le.90) string(i:i) = char(mark + 32)
 10   continue
c
      return
      end
c
c----------------------------------------------------------------------
c
      subroutine abend(iout, msg, lineno)
c
      character*(*) msg
c
      if (lineno.gt.0) then
        write (iout, '(/1x, a, '' in line no '', i3.3, ''.'')')
     .        msg(1:index(msg, '$') - 1), lineno
      else
        write (iout, '(/1x, a)') msg(1:index(msg, '$') - 1)
      endif
c
      stop '--- spins aborted.'
      end
c
c----------------------------------------------------------------------
c
      subroutine str2f(field, fnum, iout, lcount)
c
c---- Convert character field to a floating point number
c
      implicit real*8  (a-h, o-z)
      character*(*) field
      character*1 xchar(15) /'0','1','2','3','4','5','6','7','8','9',
     .                       '+','-','.','e','d'/
      logical dotrep
      data ten /10.0d0/
c
      fnum = 0.0d0
      i1 = 1
      i2 = index(field, '$') - 1
      ie2 = i2
c---- Sign
      isign = 1
      if (field(i1:i1).eq.xchar(12)) isign = -1
      if (field(i1:i1).eq.xchar(12).or.field(i1:i1).eq.xchar(11))
     .  i1 = i1 + 1
c---- Exponent
      do 10 ie = i1, i2
        if (field(ie:ie).eq.xchar(14).or.field(ie:ie).eq.xchar(15))
     .    go to 20
10    continue
      iexp = 0
      go to 50
20    i2 = ie - 1
      iexp = 1
      ie1 = ie + 1
      if (field(ie1:ie1).eq.xchar(12)) iexp = -1
      if (field(ie1:ie1).eq.xchar(12).or.field(ie1:ie1).eq.xchar(11))
     .  ie1 = ie1 + 1
      ipower = 0
      do 40 i = ie1, ie2
        do 30 j = 1, 10
          if (field(i:i).eq.xchar(j)) then
            ipower = ipower*10 + j - 1
            go to 40
          endif
 30     continue
        go to 100
 40   continue
      iexp = iexp*ipower
c---- The number itself
 50   dotrep = .false.
      do 90 i = i1, i2
        if (field(i:i).eq.xchar(13)) then
          if (dotrep) go to 100
          iexp = iexp + i - i2
          dotrep = .true.
        else 
          do 60 j = 1, 10
            if (field(i:i).eq.xchar(j)) then
              fnum = fnum*ten + dble(j-1)
              go to 90
            endif 
 60       continue
          go to 100
        endif
 90   continue
      fnum = fnum*dble(isign)*ten**iexp
      return
c
100   call abend(iout,
     .  'Illegal character when reading floating point number$', lcount)
c
      end
c
c----------------------------------------------------------------------
c
      subroutine str2i(field, inum, iout, lcount)
c
c---- Convert character field to an integer
c
      implicit real*8 (a-h, o-w)
      character*(*) field
      character*1 xchar(12) /'0','1','2','3','4','5','6','7','8','9',
     .                       '+','-'/, xtmp
c
      inum = 0
      ipower = 1
      do 150 i = index(field, '$') - 1, 1, -1
        xtmp = field(i:i)
        do 110 j = 1, 12
          if (xchar(j).eq.xtmp) go to 130
 110    continue
 120    call abend(iout,
     .    'Illegal character when reading integer$', lcount)
        stop
 130    if (j.lt.11) go to 140
        if (i.ne.1)  go to 120
        if (j.eq.12) inum = -inum
        return
140     inum = inum + (j - 1)*ipower
        ipower = ipower * 10
150   continue
c
      end
c
c----------------------------------------------------------------------
c
      subroutine getbas(basis, inp, iout, field, lcount, inpbas)
c
c---- Note: Rumer_VB means Rumer basis with VB signs of the LTs
c---- (only for input).
c
      character*(*) basis, field
      logical inpbas
c
      call getfld(inp, iout, field, lcount)
      if (index(field, 'kotani').eq.1) then
        basis = 'Kotani$'
      elseif (inpbas.and.index(field, 'rumer_vb').eq.1) then
        basis = 'Rumer_VB$'
      elseif (index(field, 'rumer').eq.1) then
        basis = 'Rumer$'
      elseif (index(field, 'serber').eq.1) then
        basis = 'Serber$'
      else
        call abend(iout, 'Unknown spin basis$', lcount)
      endif
c
      return
      end
c
c----------------------------------------------------------------------
c
      subroutine renorm(ifns, cs, rr, intype, kw)
c
      implicit real*8 (a-h, o-z)
      character*(*) intype
      dimension cs(*), rr(*), kw(*)
c
      cnorm = 0d0
      if (index(intype, 'Kotani').eq.1.or.
     .    index(intype, 'Serber').eq.1) then
        do 10 k = 1, ifns
 10     cnorm = cnorm + cs(k)*cs(k)
      elseif (index(intype, 'Rumer').eq.1) then
        do 30 k = 1, ifns
          kk = kw(k)
          rrcsk = 0d0
          do 21 l = 1, k
 21       rrcsk = rrcsk + rr(l + kk)*cs(l)
          do 22 l = k + 1, ifns
 22       rrcsk = rrcsk + rr(k + kw(l))*cs(l)
          cnorm = cnorm + cs(k)*rrcsk
 30     continue
      endif
      cnorm = 1d0/dsqrt(cnorm)
      do 40 k = 1, ifns
 40   cs(k) = cs(k)*cnorm
c
      return
      end
c
c----------------------------------------------------------------------
c
      subroutine leadt(n, spin, npairs, ifns, lt, lts, prnlt, iout)
c
c---- Generates all leading terms (LTs) in standard order
c---- Algorithm (see e.g. J. C. Manley and J. Gerratt, Comput. Phys.
c---- Commun. 31 (1984) 75-78):
c---- We start from the first LT: abab ...abaa ...a, where the number of
c---- pairs equals the number of beta spins. LT(i + 1) is obtained from
c---- LT(i) by shifting one place to the right the first beta (reading
c---- from left to right) which is followed by an alpha and by bringing
c---- all betas preceding the shifted one to the positions they held
c---- in LP(1).
c---- Also determines the VB-style signs of the LTs.
c
      implicit real*8 (a-h, o-z)
      character*(*) lt(*)
      integer*2 lts(*)
      logical prnlt
c
      npairs = (n - idint(2d0*spin))/2
      do 10 i = 1, npairs
 10   lt(1)(2*i - 1:2*i) = 'ab'
      do 20 i = 2*npairs + 1, n
 20   lt(1)(i:i) = 'a'
c
      ifns = 1
 30   i = index(lt(ifns)(1:n), 'ba')
      if (i.ne.0) then
        ifns = ifns + 1
        lt(ifns) = lt(ifns - 1)
        lt(ifns)(i:i + 1) = 'ab'
        nbeta = 0
        do 40 k = 1, i - 1
          if (lt(ifns)(k:k).eq.'b') nbeta = nbeta + 1
 40     continue
        if (nbeta.gt.0) then
          lt(ifns)(1:2*nbeta) = lt(1)(1:2*nbeta)
          do 50 j = 2*nbeta + 1, i - 1
 50       lt(ifns)(j:j) = 'a'
        endif
        go to 30
      endif
c
      do 70 i = 1, ifns 
        ltsign = 1
        nalpha = 0
        do 60 k = 1, n
          if (lt(i)(k:k).eq.'a') then
            nalpha = nalpha + 1
            if (mod(k - nalpha, 2).eq.1) ltsign = -ltsign
          endif
 60     continue
        lts(i) = ltsign 
 70   continue  
c
      if (prnlt) then
        write (iout, '(/1x, ''Leading terms (VB signs):''/)')
        do 80 i = 1, ifns
 80     write (iout, '(1x, i3, '':'', 2x, a, 
     .        3x, ''('', i2, '')'')') i, lt(i)(1:n), lts(i)
      endif
c
      return
      end
c
c----------------------------------------------------------------------
c
      subroutine fpairs(n, ifns, npairs, lt, rf)
c
c---- Forms the Rumer spin functions from the leading terms
c---- using insertion of parentheses. The Rumer functions
c---- are recorded in the integer*2 array rf(n, ifns): first
c---- the pairs and then the unpaired electrons, if any.
c
      implicit real*8 (a-h, o-z)
      character*(*) lt(*)
      integer*2 rf(n, *)
c
      do 30 l = 1, ifns
        k = 1
        kunp = 2*npairs
        do 20 i = 1, n
          if (lt(l)(i:i).eq.'a') then
            leftp = 0
            do 10 j = i + 1, n
              if (lt(l)(j:j).eq.'a') then
                leftp = leftp + 1
              elseif (lt(l)(j:j).eq.'b') then
                if (leftp.gt.0) then
                  leftp = leftp - 1
                else
                  rf(k, l) = i
                  rf(k + 1, l) = j
                  k = k + 2
                  go to 20
                endif
              endif
 10         continue
            kunp = kunp + 1
            rf(kunp, l) = i
          endif
 20     continue
 30   continue
c
      return
      end
c
c----------------------------------------------------------------------
c
      subroutine fsrb(n, ifns, lt, srb, prnks, iout)
c
c---- Maps the Serber spin functions to the Kotani spin functions
c
c     S + 1
c           \ D      A: S     ---> S using the singlet function
c       A,C  \
c     S ====> S      B: S - 1 ---> S using the triplet functions
c            /       C: S     ---> S using the triplet functions
c           / B      D: S + 1 ---> S using the triplet functions
c     S - 1
c
c     The rules are: ab ---> A
c                    aa ---> B
c                    ba ---> C
c                    bb ---> D
c
      implicit real*8 (a-h, o-z)
      character*(*) lt(*), srb(*)
      logical prnks
      character*2 wk, ws, wt
c
      nh = n/2
      do 20 j = 1, ifns
        do 10 i = 1, nh
          if (lt(ifns + 1 - j)(2*i - 1:2*i - 1).eq.'a') then
            if (lt(ifns + 1 - j)(2*i:2*i).eq.'b') then
              srb(j)(i:i) = 'A'
            else
              srb(j)(i:i) = 'B'
            endif
          else
            if (lt(ifns + 1 - j)(2*i:2*i).eq.'a') then
              srb(j)(i:i) = 'C'
            else
              srb(j)(i:i) = 'D'
            endif
          endif
 10     continue
        if (2*nh.ne.n) srb(j)(nh + 1:nh + 1) = lt(ifns + 1 - j)(n:n)
 20   continue
c
      if (prnks) then
        write (iout, '(/1x, ''Kotani and Serber spin functions---'',
     .               ''paths on the branching diagrams:''/)')
        nk = max0(n, len('Kotani'))
        ns = max0((n + 1)/2, len('Serber'))
        write (wk, '(i2)') nk
        write (ws, '(i2)') ns
        write (iout, '(1x, a3, 4x, a' // wk // ', 3x, a' // ws //
     .               ')') 'k', 'Kotani', 'Serber'
        write (wt, '(i2)') 3 + 4 + nk + 3 + ns
        write (iout, '(1x, ' // wt // '(''-''))')
        do 30 i = 1, ifns
 30     write (iout, '(1x, i3, '':'', 3x, a' // wk // ', 3x, a' // ws //
     .               ')')
     .               i, lt(ifns + 1 - i)(1:n), srb(i)(1:(n + 1)/2)
      endif
c
      return
      end
c
c----------------------------------------------------------------------
c
      subroutine cout(iout, intype, n, ifns, npairs, rf, lt, srb,
     .                c, rr, rrinv, wg, kw, prnc)
c
c---- Prints the standard designations for the Rumer, Kotani
c---- and Serber spin functions and, if requested, the vector
c---- of spin-coupling coefficients and their weights
c
      implicit real*8 (a-h, o-z)
      character*(*) intype, lt(*), srb(*)
      character*2 pstr, ostr
      character*255 line
      character*2 ws, wt
      integer*2 rf(n, ifns)
      logical prnc
      dimension c(*), rr(*), rrinv(*), wg(*), kw(*)
c
      if (.not.prnc)
     .  write (iout, '(/1x, a, '' spin functions:''/)')
     .  intype(1:index(intype, '$') - 1)
c
      if (index(intype, 'Rumer').eq.1) then
        if (prnc) then
c-------- Start by calculating Gallups' weights (G.A. Gallup, 
c-------- R.L.Vance, J.R. Collins and J.M. Norbeck,
c-------- Adv. Quantum Chem. 16 (1982) 229). 
          wnorm = 0d0
          do 1 l = 1, ifns
            wgl = c(l)*c(l)/rrinv(kw(l + 1))
            wg(l) = wgl       
            wnorm = wnorm + wgl
 1        continue
          do 2 l = 1, ifns
 2        wg(l) = wg(l)/wnorm   
        endif
        write (pstr, '(i2)') npairs
        write (ostr, '(i2)') n - 2*npairs
        do 20 l = 1, ifns
          if (n.eq.2*npairs) then
            write (line, '('// pstr //
     .            '(''('', i2, '' - '', i2, '')''))')
     .            (rf(k, l), k = 1, n)
          elseif (npairs.eq.0) then
            write (line, '('// ostr // '(''('', i2, '')''))')
     .            (rf(k, l), k = 1, n)
          else
            write (line, '('// pstr //
     .            '(''('', i2, '' - '', i2, '')''),'
     .            // ostr // '(''('', i2, '')''))')
     .            (rf(k, l), k = 1, n)
          endif
 9        i = index(line, ')(')
          if (i.ne.0) then
            line = line(:i - 1) // ',' // line(i + 2:)
            go to 9
          endif
 10       i = index(line, ' ')
          if (index(line(i + 1:), ')').ne.0) then
            line = line(:i - 1) // line(i + 1:)
            go to 10
          endif
          if (l.eq.1) then
            ns = max0(i - 1, len('Spin-coupling pattern'))
            write (ws, '(i2)') ns
          endif
          if (prnc) then
            if (l.eq.1) then
              write (iout, '(1x, a3, 3x, a' // ws // ', (2x, a12),
     .          2(2x, a9))')
     .          'k', 'Spin-Coupling pattern', 'Coefficient', 'Weight',
     .          'GWeight'
              nt = 3 + 3 + ns + (2 + 12) + 2*(2 + 9)
              write (wt, '(i2)') nt
              write (iout, '(1x, ' // wt // '(''-''))')
            endif
            ll = kw(l)
            wl = 0d0
            do 11 m = 1, l
 11         wl = wl + rr(m + ll)*c(m)
            do 12 m = l + 1, ifns
 12         wl = wl + rr(l + kw(m))*c(m)
            wl = c(l)*wl
            write (iout, '(1x, i3, '':'', 2x, a' // ws // ',
     .            (2x, f12.6), 2(2x, f9.6))') 
     .            l, line(1:i - 1), c(l), wl, wg(l)
          else
            if (l.eq.1) then
              write (iout, '(1x, a3, 3x, a' // ws // ')')
     .          'k', 'Spin-Coupling pattern'
              nt = 3 + 3 + ns
              write (wt, '(i2)') nt
              write (iout, '(1x, ' // wt // '(''-''))')
            endif
            write (iout, '(1x, i3, '':'', 2x, a' // wt // ')')
     .            l,  line(1:i - 1)
          endif
 20     continue
        return
      endif
c
      if (index(intype, 'Kotani').eq.1) then
        do 40 i = 1, ifns
          inom = 0
          ipos = 1
          line = '('
          do 30 j = 1, n - 1
            if (lt(ifns + 1 - i)(j:j).eq.'a') then
              inom = inom + 1
            else
              inom = inom - 1
            endif
            if ((inom/2)*2.eq.inom) then
              write (line(ipos + 1:), '(i2)') inom/2
              ipos = ipos + 2
              if (line(ipos - 1:ipos - 1).eq.' ') then
                line = line(1:ipos - 2) // line(ipos:ipos)
                ipos = ipos - 1
              endif
            else
              write (line(ipos + 1:), '(''('', i2, ''/2)'')') inom
              ipos = ipos + 6
              if (line(ipos - 4:ipos - 4).eq.' ') then
                line = line(1:ipos - 5) // line(ipos - 3:)
                ipos = ipos - 1
              endif
            endif
 30       continue
          line = line(1:ipos) // ')'
          ipos = ipos + 1
          if (i.eq.1) then
            ns = max0(ipos, len('Spin-coupling pattern'))
            write (ws, '(i2)') ns
          endif
          if (prnc) then
            if (i.eq.1) then
              write (iout, '(1x, a3, 3x, a' // ws // ', 2(2x, a12))')
     .          'k', 'Spin-Coupling pattern', 'Coefficient', 'Weight'
              nt = 3 + 3 + ns + 2*(2 + 12)
              write (wt, '(i2)') nt
              write (iout, '(1x, ' // wt // '(''-''))')
            endif
            write (iout, '(1x, i3, '':'', 2x, a' // ws // ',
     .            2(2x, f12.6))') i, line(1:ipos), c(i), c(i)*c(i)
          else
          if (i.eq.1) then
              write (iout, '(1x, a3, 3x, a' // ws // ')')
     .          'k', 'Spin-Coupling pattern'
              nt = 3 + 3 + ns
              write (wt, '(i2)') nt
              write (iout, '(1x, ' // wt // '(''-''))')
            endif
            write (iout, '(1x, i3, '':'', 2x, a' // ws // ')')
     .            i, line(1:ipos)
          endif
 40     continue
        return
      endif
c
      if (index(intype, 'Serber').eq.1) then
        do 60 i = 1, ifns
          if (srb(i)(1:1).eq.'A') then
            ispin = 0
            line(1:1) = '0'
          else
            ispin = 1
            line(1:1) = '1'
          endif
          ipos = 1
          do 50 j = 2, (n + 1)/2 - 1
            if (srb(i)(j:j).eq.'B') then
              ispin = ispin + 1
            elseif (srb(i)(j:j).eq.'D') then
              ispin = ispin - 1
            endif
            if (srb(i)(j:j).eq.'A') then
              line(ipos + 1:ipos + 2) = '0)'
            else
              line(ipos + 1:ipos + 2) = '1)'
            endif
            ipos = ipos + 2
            write (line(ipos + 1:), '(i2)') ispin
            ipos = ipos + 2
            if (line(ipos - 1:ipos - 1).eq.' ') then
              line = line(1:ipos - 2) // line(ipos:ipos)
              ipos = ipos - 1
            endif
            do 45 k = ipos, 1, -1
 45         line(k + 1:k + 1) = line(k:k)
            line(1:1) = '('
            line(ipos + 2:ipos + 2) = ';'
            ipos = ipos + 2
 50       continue
	  do 55 k = ipos, 1, -1
 55       line(k + 1:k + 1) = line(k:k)
          line(1:1) = '('
          if ((n/2)*2.eq.n) then
            if (srb(i)(n/2:n/2).eq.'A') then
              line(ipos + 2:ipos + 3) = '0)'
            else
              line(ipos + 2:ipos + 3) =  '1)'
            endif
            ipos = ipos + 3
          else
            line(ipos + 2:ipos + 7) = '(1/2))'
            ipos = ipos + 7
          endif
          if (i.eq.1) then
            ns = max0(ipos, len('Spin-coupling pattern'))
            write (ws, '(i2)') ns
          endif
          if (prnc) then
            if (i.eq.1) then
              write (iout, '(1x, a3, 3x, a' // ws // ', 2(2x, a12))')
     .          'k', 'Spin-Coupling pattern', 'Coefficient', 'Weight'
              nt = 3 + 3 + ns + 2*(2 + 12)
              write (wt, '(i2)') nt
              write (iout, '(1x, ' // wt // '(''-''))')
            endif
            write (iout, '(1x, i3, '':'', 2x, a' // ws // ',
     .            2(2x, f12.6))') i, line(1:ipos), c(i), c(i)*c(i)
          else
          if (i.eq.1) then
              write (iout, '(1x, a3, 3x, a' // ws // ')')
     .          'k', 'Spin-Coupling pattern'
              nt = 3 + 3 + ns
              write (wt, '(i2)') nt
              write (iout, '(1x, ' // wt // '(''-''))')
            endif
            write (iout, '(1x, i3, '':'', 2x, a' // ws // ')')
     .            i, line(1:ipos)
          endif
 60     continue
        return
      endif
c
      end
c
c----------------------------------------------------------------------
c
      subroutine fdeter(n, ifns, npairs, ndet, rf, rd, rs, prndet, iout)
c
c---- Expands the Rumer spin functions in terms of signed spin factors
c---- The determinants are recorded in the integer*4 array rd(ndet, ifns)
c---- and their signs---in the integer*2 array rs(ndet, ifns). Alpha's
c---- are bit-coded as cleared bits (0), and beta's---as set bits (1) 
c
      implicit real*8 (a-h, o-z)
      integer*4 rd(ndet, *)
      character*1 pstr, signs(-1:1) /'-', 2*' '/ ! MS DOS      
      character*2 nstr ! MS DOS
      character*80 fstr
      character*31 rd2str
      integer*2 rf(n, *), rs(ndet, *)
      logical prndet
c
      do 40 l = 1, ifns
        do 5 i = 1, ndet
          rs(i, l) = 1
          rd(i, l) = 0
  5     continue  
        do 30 ipair = 1, npairs
          ip = rf(2*ipair - 1, l)
          jp = rf(2*ipair, l)
          lpart = 2**(npairs - ipair)
          do 20 j = 1, 2**(ipair - 1)
            joffs = (j - 1)*2*lpart
            do 10 k = joffs + 1, joffs + lpart
              rd(k, l) = ibset(rd(k, l), jp)
              klp = k + lpart
              rd(klp, l) = ibset(rd(klp, l), ip) 
              rs(klp, l) = -rs(klp, l)
 10         continue
 20       continue
 30     continue
 40   continue
c
      if (prndet) then
        write (iout,
     .      '(/1x, ''Rumer functions expanded in terms of '',
     .             ''signed spin factors:'')')
        if (n.le.8) then
          pstr = '4'  ! MS DOS
        else
          pstr = '3'  ! MS DOS
        endif 
        write (nstr, '(i2)') n
        fstr = '((1x, ' // pstr //
     .         '(i3, '':'', 1x, a, ''('', a' // nstr // ', '')'', 1x)))'
        do 50 l = 1, ifns
          write (iout, '(/1x, ''Spin function '', i3/)') l
          write (iout, fstr) (k, signs(rs(k, l)), 
     .                           rd2str(n, rd(k, l)), k = 1, ndet)       
 50     continue
      endif
c
      return
      end
c
c----------------------------------------------------------------------
c
      character*31 function rd2str(n, rd)
c
c---- Converts a bit-coded spin-product to a string
c      
      integer*4 rd
c
      do 10 i = 1, n
        if (btest(rd, i)) then
          rd2str(i:i) = 'b'
        else      
          rd2str(i:i) = 'a'
        endif
 10   continue
c
      return
      end  
c
c----------------------------------------------------------------------
c
      subroutine rover(ifns, npairs, ndet, rd, rs, rr, rrinv,
     .                 kpvt, work, prnovr, iout)
c
c---- Calculates the overlap matrix between the Rumer spin functions
c
      implicit real*8 (a-h, o-z)
      integer*4 rd(ndet, *), rdki
      integer*2 rs(ndet, *), rski
      logical prnovr
      dimension rr(*), rrinv(*), kpvt(*), work(*), det(2), inert(3)
c
      factor = 1d0/dfloat(2**npairs)
      ij = 0
      do 20 j = 1, ifns
      do 20 i = 1, j
        ij = ij + 1
        if (i.eq.j) then
          rr(ij) = 1d0
        else
          irr = 0
          do 15 k = 1, ndet
            rdki = rd(k, i)
            rski = rs(k, i)
            do 10 l = 1, ndet
              if (rdki.eq.rd(l, j)) irr = irr + rski*rs(l, j)
 10         continue
 15       continue   
          rr(ij) = factor*dfloat(irr)
        endif
 20   continue
c
      if (prnovr) then
        write (iout, '(/1x,
     .        ''Overlap matrix b/n the Rumer functions:''/)')
        call utout(iout, rr, ifns, 1, 5)
      endif
c
c---- Invert the overlap matrix between the Rumer functions ------
c
      do 30 kl = 1, ifns*(ifns + 1)/2
  30  rrinv(kl) = rr(kl)
      call dspfa(rrinv, ifns, kpvt, info)
      if (info.ne.0) call abend(iout,
     .  '(''< R | R > is singular. Cannot continue ...$'')', 0)
      call dspdi(rrinv, ifns, kpvt, det, inert, work, 001)
c
      return
      end
c
c----------------------------------------------------------------------
c
      subroutine rumkot(ifns, rr, trk, a, kw, prntr, iout)
c
c---- Calculates the Rumer -> Kotani transformation matrix
c---- by Schmidt orthonormalization
c
      implicit real*8 (a-h, o-z)
      dimension rr(*), trk(*), a(*), kw(*)
      logical prntr
c
      do 90 i = 1, ifns
        ic = kw(i)
        trk(i + ic) = 1d0
        if (i.gt.1) then
          do 10 k = 1, i - 1
 10       trk(k + ic) = 0d0
          do 30 j = 1, i - 1
            jc = j*(j - 1)/2
            aj = 0d0
            do 20 l = 1, j
 20         aj = aj + trk(l + jc)*rr(l + ic)
            a(j) = aj
 30       continue
          do 50 j = 1, i - 1
            jc = kw(j)
            do 40 k = 1, j
 40         trk(k + ic) = trk(k + ic) - trk(k + jc)*a(j)
 50       continue
          t1 = 0d0
          t2 = 0d0
          kl = 0
          do 70 l = 1, i
            trkli = trk(l + ic)
            do 60 k = 1, l - 1
              kl = kl + 1
              t2 = t2 + trk(k + ic)*trkli*rr(kl)
 60         continue
            kl = kl + 1
            t1 = t1 + trkli*trkli*rr(kl)
 70       continue
          t = 1d0/dsqrt(t1 + t2 + t2)
          do 80 k = 1, i
 80       trk(k + ic) = t*trk(k + ic)
        endif
 90   continue
c
      if (prntr) then
        write (iout, '(/1x, ''Rumer to Kotani '',
     .    ''(K_{f^N_S}, ..., K_1) = (R_1, ..., R_{f^N_S}) T^{RK}:''/)')
        call utout(iout, trk, ifns, 1, 5)
      endif
c
      return
      end
c
c----------------------------------------------------------------------
c
      subroutine kotrum(ifns, trk, tkr, tkrsq, prntr, iout)
c
c---- Calculates the Kotani -> Rumer transformation matrix
c
      implicit real*8 (a-h, o-z)
      logical prntr
      dimension trk(*), tkr(*), tkrsq(ifns, *)
c
      ij = 0
      do 10 j = 1, ifns
      do 10 i = 1, j
        ij = ij + 1
        tkrsq(i, j) = trk(ij)
 10   continue
      call dtrdi(tkrsq, ifns, ifns, det, 011, info)
      if (info.ne.0) call abend(iout,
     .  '(''T^{RK} is singular. Cannot continue ...$'')', 0)
      ij = 0
      do 20 j = 1, ifns
      do 20 i = 1, j
        ij = ij + 1
        tkr(ij) = tkrsq(i, j)
 20   continue
c
      if (prntr) then
        write (iout, '(/1x, ''Kotani to Rumer '',
     .    ''(K_{f^N_S}, ..., K_1) T^{KR} = (R_1, ..., R_{f^N_S}):''/)')
        call utout(iout, tkr, ifns, 1, 5)
      endif
c
      return
      end
c
c----------------------------------------------------------------------
c
      subroutine kotsrb(n, ifns, srb, tks, is, prntr, iout)
c
c---- Calculates the Kotani to Serber transformation matrix
c
      implicit real*8 (a-h, o-z)
      character*(*) srb(*)
      logical prntr
      dimension tks(ifns, *), is(*)
c
c---- Initialize variables
c
      nh = n/2
      do 20 j = 1, ifns
        if (srb(j)(1:1).eq.'A') then
          is(j) = 0
        else
          is(j) = 1
        endif
        tks(j, j) = 1d0
        do 10 i = 1, j - 1
          tks(i, j) = 0d0
          tks(j, i) = 0d0
 10     continue
 20   continue
c
c---- Calculate the transformaiton matrix
c
      do 100 i = 2, nh
c------ Update the partial spins
        do 30 j = 1, ifns
          if (srb(j)(i:i).eq.'B') then
            is(j) = is(j) + 1
          elseif (srb(j)(i:i).eq.'D') then
            is(j) = is(j) - 1
          endif
 30     continue
c------ Search for a region of equal partial spins
        j2 = 1
 40     j1 = j2
        j2 = j1 + 1
 50     if (j1.ge.ifns) go to 100
        if (is(j1).ne.is(j2)) then
          if (j2 - j1.gt.1) goto 60
          j1 = j1 + 1
        endif
        j2 = j2 + 1
        go to 50
c------ Search for A's followed by (an equal number of) C's
 60     j = j1
        jend = j2 - 1
 70     if (srb(j)(i:i).ne.'A') then
          if (j.eq.jend) go to 40
          j = j + 1
          go to 70
        endif
        i0 = j
        if (j.eq.jend) go to 40
        j = j + 1
 80     if (srb(j)(i:i).eq.'A') then
          if (j.eq.jend) go to 40
          j = j + 1
          go to 80
        endif
        n0 = j - i0
        if (srb(j)(i:i).ne.'C') go to 40
c------ Transform the partial spin functions
        s = dfloat(is(j1))
        x = dsqrt((s + 1d0)/(2d0*s + 1d0))
        y = dsqrt(s/(2d0*s + 1d0))
        do 90 k = i0, i0 + n0 - 1
        do 90 l = i0, i0 + n0 - 1
          xtkl = x*tks(k, l)
          ytkl = y*tks(k, l)
          tks(k, l) = xtkl
          tks(k + n0, l) = -ytkl
          tks(k, l + n0) = ytkl
          tks(k + n0, l + n0) = xtkl
  90    continue
        go to 40
 100  continue
c
      if (prntr) then
        write (iout, '(/1x, ''Kotani to Serber '',
     .    ''(S_1, ..., S_{f^N_S}) = (K_1, ..., K_{f^N_S}) T^{KS}:''/)')
        call fulout(iout, tks, tks, ifns, ifns, 2, 5)
      endif
c
      return
      end
c
c----------------------------------------------------------------------
c
      subroutine permr(n, ifns, ndet, npairs, iperm, rd, rdp, rs,
     .                 rrinv, rpr, vr, kw)
c
c---- Calculates an S_N representation matrix in the Rumer basis
c
      implicit real*8 (a-h, o-z)
      integer*4 rd(ndet, *), rdp(ndet, *), det, detp, rdki
      integer*2 rs(ndet, *), rski
      dimension iperm(*), rrinv(*), rpr(ifns, *), vr(ifns, *), kw(*)
c
c---- Permute the determinants
c
      do 20 i = 1, ifns
      do 20 k = 1, ndet
        det = rd(k, i)
        detp = 0
        do 10 j = 1, n
          jp = iperm(j)
          if (btest(det, jp)) detp = ibset(detp, j)
 10     continue
        rdp(k, i) = detp
 20   continue
c
c---- Calculate < R | P | R >
c
      factor = 1d0/dfloat(2**npairs)
      do 40 i = 1, ifns
      do 40 j = 1, ifns
        irpr = 0
        do 35 k = 1, ndet
          rdki = rd(k, i)
          rski = rs(k, i)
          do 30 l = 1, ndet
            if (rdki.eq.rdp(l, j)) irpr = irpr + rski*rs(l, j)
 30       continue
 35     continue  
        rpr(i, j) = factor*dfloat(irpr)
 40   continue
c
c---- Calculate V^R(P) = < R | R >^{-1} < R | P | R >
c
      do 70 i = 1, ifns
        ic = kw(i)
        do 60 j = 1, ifns
          vrij = 0d0
          do 51 k = 1, i
 51       vrij = vrij + rrinv(k + ic)*rpr(k, j)
          do 52 k = i + 1, ifns
 52       vrij = vrij + rrinv(i + kw(k))*rpr(k, j)
          vr(i, j) = vrij
 60     continue
 70   continue
c
      return
      end
c
c----------------------------------------------------------------------
c
      subroutine permk(ifns, vr, trk, tkr, vk, w, kw)
c
c---- Calculates an S_N representation matrix in the Kotani basis
c---- (call permr first)
c
      implicit real*8 (a-h, o-z)
      dimension vr(ifns, *), trk(*), tkr(*), vk(ifns, *), w(*), kw(*)
c
c---- Calculate V^{K}(P) = T^{KR} V^{R}(P) T^{RK}
c
      do 50 j = 1, ifns
        jc = kw(j)
        jfns = ifns + 1 - j
        do 20 k = 1, ifns
          wk = 0d0
          do 10  l = 1, j
 10       wk = wk + vr(k, l)*trk(l + jc)
          w(k) = wk
 20     continue
        do 40 i = 1, ifns
          vkij = 0d0
          do 30 k = i, ifns
 30       vkij = vkij + tkr(i + kw(k))*w(k)
          vk(ifns + 1 - i, jfns) = vkij
 40     continue
 50   continue
c
      return
      end
c
c----------------------------------------------------------------------
c
      subroutine perms(ifns, vk, tks, vs, w)
c
c---- Calculates an S_N representation matrix in the Serber basis
c---- (call permr and permk first)
c
      implicit real*8 (a-h, o-z)
      dimension vk(ifns, *), tks(ifns, *), vs(ifns, *), w(*)
c
c---- Calculate V^{S}(P) = T^{SK} V^{K}(P) T^{KS}
c
      do 50 j = 1, ifns
        do 20 k = 1, ifns
          wk = 0d0
          do 10 l = 1, ifns
  10      wk = wk + vk(k, l)*tks(l, j)
          w(k) = wk
  20    continue
        do 40 i = 1, ifns
          vsij = 0d0
          do 30 k = 1, ifns
 30       vsij = vsij + tks(k, i)*w(k)
          vs(i, j) = vsij
 40     continue
 50   continue
c
      return
      end
c
c----------------------------------------------------------------------
c
      subroutine matvec(n, ac, a, c, kw, kode)
c
      implicit real*8 (a-h, o-z)
      dimension ac(*), a(*), c(*), kw(*)
c
c---- If kode=1 c = ac, if kode=2 c = (a^T)c,
c---- if kode=3 a is packed upper-triangular and ac is in inverse order
c---- if kode=4 a is packed upper-triangular and c is in inverse order
c
      if (kode.eq.1) then
        do 20 i = 1, n
          aci = 0d0
          jofs = 0
          do 10 j = 1, n
            aci = aci + a(i + jofs)*c(j)
            jofs = jofs + n
 10       continue
          ac(i) = aci
 20     continue
      elseif (kode.eq.2) then
        iofs = 0
        do 40 i = 1, n
          aci = 0d0
          do 30 j = 1, n
 30       aci = aci + a(j + iofs)*c(j)
          ac(i) = aci
          iofs = iofs + n
 40     continue
      elseif (kode.eq.3) then
        np1 = n + 1
        do 60 i = 1, n
          aci = 0d0
          do 50 j = i, n
 50       aci = aci + a(i + kw(j))*c(j)
          ac(np1 - i) = aci
 60     continue
      elseif (kode.eq.4) then
        np1 = n + 1
        do 80 i = 1, n
          aci = 0d0
          do 70 j = i, n
 70       aci = aci + a(i + kw(j))*c(np1 - j)
          ac(i) = aci
 80     continue
      endif
c
      do 90 i = 1, n
 90   c(i) = ac(i)
c
      return
      end
c
c----------------------------------------------------------------------
c
      subroutine timer(utime, stime)
c     
c---- uses etime
c
      real*4 tarray(2), elapse
c
      elapse = etime(tarray)
      utime = tarray(1)
      stime = tarray(2)
c
      return
      end
